<?php
/**
 * Sitemap Stylesheet Class
 *
 * Handles the generation of XML stylesheets for sitemaps.
 *
 * @package SureRank
 * @since 0.0.1
 */

namespace SureRank\Inc\Sitemap;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Stylesheet
 *
 * Generates sitemap stylesheets dynamically.
 */
class Stylesheet {

	/**
	 * Generates the XML for the sitemap stylesheet.
	 *
	 * @param string $sitemap_title The title of the sitemap.
	 * @param string $sitemap_slug  The slug of the sitemap.
	 * @return string The generated XML stylesheet as a string.
	 */
	public function generate( string $sitemap_title, string $sitemap_slug ): string {
		ob_start();
		echo '<?xml version="1.0" encoding="UTF-8"?>'; // Direct echo to avoid PHP parsing issues.
		?>
		<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:s="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">
			<xsl:output method="html" encoding="UTF-8" indent="yes" />
			<xsl:template match="/">
				<html>
				<head>
					<meta http-equiv="Content-Type" content="text/html; charset=UTF-8" />
					<title><?php echo esc_html( $sitemap_title ); ?></title>
					<link href="https://fonts.googleapis.com/css2?family=Figtree:ital,wght@0,300..900;1,300..900&amp;display=swap" rel="stylesheet" />
					<style>
						body {
							font-family: Figtree, sans-serif;
							background-color: #fff;
							color: #333;
							line-height: 1.6;
							margin: 0;
						}
						h1, h2 {
							color: #fff;
							margin: 0px;
							margin-bottom: 8px;
						}
						p {
							color: #E5E7EB;
							margin: 0px;
							font-size: 14px;	
							font-weight: 400;
						}
						table {
							width: 100%;
							border-collapse: separate;
							border-spacing: 0;
							background-color: #fff;
							border-radius: 8px;
							box-shadow: 0px 4px 8px rgba(0, 0, 0, 0.05);
							border: 1px solid #E5E7EB;				
						}
						th, td {
							padding: 12px;
							text-align: left;

						}
						th {
							background-color: #F9FAFB;
							color: #111827;
							font-weight: 500;
							font-size: 14px;
							border-radius: 8px 8px 0 0;

						}
						td {
							background-color: #fdfdfd;
							color: #4B5563;
							padding: 22px 12px;
							font-size: 14px;
							font-weight: 400;
							border-top: 1px solid #E5E7EB;
							border-radius: 0px 0px 8px 8px;

						}
						td a {
							color: #4330D2;
							text-decoration: none;
							font-weight: 400;
						}
						a {
							color: #E5E7EB;
							text-decoration: underline;
							font-size: 14px;
							font-weight: 400;
						}
						a:hover {
							text-decoration: underline;
						}
						.date {
							color: #4B5563;
							font-size: 0.9em;
						}
						.image-list {
							font-style: italic;
							color: #4B5563;
						}
						ul {
							list-style-type: none;
							padding: 0;
							margin: 0;
						}
						li {
							margin-bottom: 10px;
						}
						.sitemap-container {
							background-color: #4330D2;
							padding: 32px;
						}

						.sitemap-content {
							padding: 16px;
							margin: 0 auto;
							max-width: 1100px;
							margin-top:16px;
						}
						.sitemap-count {
							font-size: 14px;
							font-weight: 400;
							color: #111827;
							padding-bottom: 16px;
						}
						.sitemap-index {
							font-size: 12px;
							font-weight: 600;
							color: #4330D2;
							text-decoration: none;
							padding-bottom: 16px;
						}
						.sitemap-index div {
							margin-bottom: 20px;
						}
					</style>
				</head>
				<body>
					<div class="sitemap-container">
						<h2><?php echo esc_html( $sitemap_title ); ?></h2>
						<p>This XML Sitemap was generated by the <a target="_blank" href="https://surerank.com">SureRank WordPress SEO Plugin</a>.  It helps search engines like Google, Bing, etc. crawl your website’s posts, pages, products, images, and archives.</p>
						<p><a target="_blank" href="https://www.sitemaps.org/protocol.html">Learn more about XML Sitemaps.</a></p>
					</div>	
					<div class="sitemap-content">
						<xsl:choose>
							<xsl:when test="s:sitemapindex">
								<p class="sitemap-count" style="font-size: 16px;"> This XML Sitemap Index file contains <b><xsl:value-of select="count(s:sitemapindex/s:sitemap)" /> sitemaps.</b></p>
								<table>
									<thead>
										<tr>
											<th>Sitemap</th>
											<th>Last Modified</th>
										</tr>
									</thead>
									<tbody>
										<xsl:for-each select="s:sitemapindex/s:sitemap">
											<tr>
												<td>
													<a href="{s:loc}">
														<xsl:value-of select="s:loc" />
													</a>
												</td>
												<td>
													<xsl:value-of select="s:lastmod" />
												</td>
											</tr>
										</xsl:for-each>
									</tbody>
								</table>
							</xsl:when>
							<xsl:when test="count(s:urlset/s:url) = 0">
								<p class="sitemap-count">No Indexable URLs found for this sitemap.</p>
								<a href="/<?php echo esc_attr( $sitemap_slug ); ?>" class="sitemap-index">
									<div style="display: flex; align-items: center; gap: 8px;">
										<svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg">
											<path d="M11.0846 7H2.91797" stroke="#4330D2" stroke-width="1.25" stroke-linecap="round" stroke-linejoin="round"/>
											<path d="M7.0013 11.0837L2.91797 7.00033L7.0013 2.91699" stroke="#4330D2" stroke-width="1.25" stroke-linecap="round" stroke-linejoin="round"/>
										</svg>
										Back to Sitemap Index
									</div>
								</a>
							</xsl:when>
							<xsl:otherwise>
								<p class="sitemap-count" style="font-size: 16px;"> This XML Sitemap contains <b><xsl:value-of select="count(s:urlset/s:url)" /></b> URLs.</p>
								<a href="/<?php echo esc_attr( $sitemap_slug ); ?>" class="sitemap-index">
									<div style="display: flex; align-items: center; gap: 8px;">
										<svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg">
											<path d="M11.0846 7H2.91797" stroke="#4330D2" stroke-width="1.25" stroke-linecap="round" stroke-linejoin="round"/>
											<path d="M7.0013 11.0837L2.91797 7.00033L7.0013 2.91699" stroke="#4330D2" stroke-width="1.25" stroke-linecap="round" stroke-linejoin="round"/>
										</svg>
										Back to Sitemap Index
									</div>
								</a>
								<table>
									<thead>
										<tr>
											<th>URL</th>
											<th>Images</th>
											<th>Last Modified</th>
										</tr>
									</thead>
									<tbody>
										<xsl:for-each select="s:urlset/s:url">
											<tr>
												<td>
													<a href="{s:loc}">
														<xsl:value-of select="s:loc" />
													</a>
												</td>
												<td class="image-list">
													<xsl:choose>
														<xsl:when test="image:image">
															<xsl:value-of select="count(image:image)" />
														</xsl:when>
														<xsl:otherwise>0</xsl:otherwise>
													</xsl:choose>
												</td>
												<td class="date">
													<xsl:value-of select="s:lastmod" />
												</td>
											</tr>
										</xsl:for-each>
									</tbody>
								</table>
							</xsl:otherwise>
						</xsl:choose>
					</div>
				</body>
				</html>
			</xsl:template>
		</xsl:stylesheet>
		<?php
		return (string) ob_get_clean();
	}
}
